/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id:  $:

  Description:
    Implemnation of the "Device/Download" config file download dialog

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2008-05-14  License download added
    2007-04-19  File selection extended with nx?
    2006-07-28  Clearing of last error and progress bar included
    2006-07-04  initial version

**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file DownloadDlg.cpp
///  Implemnation of the "Device/Download" config file download dialog
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "cifXTest.h"
#include "DownloadDlg.h"
#include "CifXDeviceBase.h"
#include "CifXTestDlg.h"

/////////////////////////////////////////////////////////////////////////////
/// List of available download modes
/////////////////////////////////////////////////////////////////////////////
static CDownloadDlg::DOWNLOAD_MODES s_atDownloadModes[] =
{
  { _T("Firmware Download"),      
    DOWNLOAD_MODE_FIRMWARE, 
    _T("netX Firmware Files (*.nxf,*.nxm,*.mod)|*.nxf; *.nxm; *.mod|All Files (*.*)|*.*||")
  },
  { _T("Configuration Download"), 
    DOWNLOAD_MODE_CONFIG,   
    _T("netX Configuration Files (*.nxd, *.dbm)|*.nxd; *.dbm|All Files (*.*)|*.*||")
  },
  { _T("File Download"),          
    DOWNLOAD_MODE_FILE,     
    _T("All Files (*.*)|*.*||")
  },
  { _T("License Download"),
    DOWNLOAD_MODE_LICENSECODE,     
    _T("netX License Files (*.nxl)|*.nxl|All Files (*.*)|*.*||")
  },
};

IMPLEMENT_DYNAMIC(CDownloadDlg, CBaseDialog)

/////////////////////////////////////////////////////////////////////////////
/// Default Constructor
///   \param pParent Parent Window
/////////////////////////////////////////////////////////////////////////////
CDownloadDlg::CDownloadDlg(CWnd* pParent /*=NULL*/)
	: CBaseDialog(CDownloadDlg::IDD, pParent, false)
  , m_ulChannel(0)
  , m_ptMode(NULL)
  , m_pabFileData(NULL)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CDownloadDlg::~CDownloadDlg()
{
  if(NULL != m_pabFileData)
  {
    delete [] m_pabFileData;
    m_pabFileData = NULL;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// DDX/DDV Support
///   \param pDX 
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::DoDataExchange(CDataExchange* pDX)
{
  CBaseDialog::DoDataExchange(pDX);
  DDX_Control(pDX, IDC_PROG_DOWNLOAD, m_cDownloadProgressCtrl);
  DDX_Control(pDX, IDC_CB_DOWNLOADMODE, m_cDownloadModeCtrl);
}

BEGIN_MESSAGE_MAP(CDownloadDlg, CBaseDialog)
  ON_CBN_SELCHANGE(IDC_CB_DOWNLOADMODE, OnCbnSelchangeCbDownloadmode)
  ON_EN_KILLFOCUS(IDC_EDT_CHANNEL, OnEnKillfocusEdtChannel)
  ON_BN_CLICKED(IDC_BTN_SELECTFILE, OnBnClickedBtnSelectfile)
  ON_BN_CLICKED(IDC_BTN_DOWNLOAD, OnBnClickedBtnDownload)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
/// First Time Dialog initialization
///   \return TRUE
/////////////////////////////////////////////////////////////////////////////
BOOL CDownloadDlg::OnInitDialog()
{
  CBaseDialog::OnInitDialog();

  for(int iIdx = 0; iIdx < sizeof(s_atDownloadModes) / sizeof(s_atDownloadModes[0]); ++iIdx)
  {
    int iItem = m_cDownloadModeCtrl.AddString(s_atDownloadModes[iIdx].szMode);
    m_cDownloadModeCtrl.SetItemDataPtr(iItem, &s_atDownloadModes[iIdx]);
  }

  m_cDownloadModeCtrl.SetCurSel(0);
  OnCbnSelchangeCbDownloadmode();

  return TRUE;  // return TRUE unless you set the focus to a control
  // EXCEPTION: OCX Property Pages should return FALSE
}

/////////////////////////////////////////////////////////////////////////////
/// Download mode selection changed
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::OnCbnSelchangeCbDownloadmode()
{
  int iSel = m_cDownloadModeCtrl.GetCurSel();
  ASSERT(-1 != iSel);

  m_ptMode = (PDOWNLOAD_MODES)m_cDownloadModeCtrl.GetItemDataPtr(iSel);

  SetDlgItemText(IDC_EDT_FILENAME, m_ptMode->csLastFullFilename);
  
  // Remove last error and clear progress bar
  SetDlgItemText(IDC_LASTERR, NULL);
  m_cDownloadProgressCtrl.SetPos(0);
}

/////////////////////////////////////////////////////////////////////////////
/// Channel number editing end event
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::OnEnKillfocusEdtChannel()
{
  CString csTemp;
  GetDlgItemText(IDC_EDT_CHANNEL, csTemp);

  TCHAR* pszEnd = NULL;
  m_ulChannel = _tcstoul(csTemp, &pszEnd, 10);

  csTemp.Format(_T("%u"), m_ulChannel);
  SetDlgItemText(IDC_EDT_CHANNEL, csTemp);
  
  // Remove last error and clear progress bar
  SetDlgItemText(IDC_LASTERR, NULL);
  m_cDownloadProgressCtrl.SetPos(0);
}

/////////////////////////////////////////////////////////////////////////////
/// Select File for download button clicked event
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::OnBnClickedBtnSelectfile()
{
  CFileDialog cFileDlg(TRUE, 
                       _T(".mod"),
                       NULL, 
                       OFN_FILEMUSTEXIST,
                       m_ptMode->szFileFilter);

  if(cFileDlg.DoModal() == IDOK)
  {
    m_ptMode->csLastShortFilename = cFileDlg.GetFileName();
    m_ptMode->csLastFullFilename  = cFileDlg.GetPathName();
    SetDlgItemText(IDC_EDT_FILENAME, m_ptMode->csLastFullFilename);
  }
  
  // Remove last error and clear progress bar
  SetDlgItemText(IDC_LASTERR, NULL);
  m_cDownloadProgressCtrl.SetPos(0);
}

/////////////////////////////////////////////////////////////////////////////
/// Progress Callback for cifX Driver
///   \param ulStep       Current step that has been performed
///   \param ulMaxStep    Maximum number of steps to do
///   \param pvUser       user specific pointer (Pointer to class instance in this case)
///   \param bFinished    != 0 to signal end of download
///   \param lError       Error that happened during download CIFX_NO_ERROR for successful download
/////////////////////////////////////////////////////////////////////////////
static void APIENTRY DownloadProgress(uint32_t ulStep, uint32_t ulMaxStep, void* pvUser, int8_t bFinished, int32_t lError)
{
  CDownloadDlg* pcDlg = (CDownloadDlg*)pvUser;

  pcDlg->UpdateProgress(ulStep, ulMaxStep, bFinished, lError);
}

/////////////////////////////////////////////////////////////////////////////
/// Thread to handle download, to avoid message loop blocking while download
///   \param pvParam Pointer to class instance
///   \return 0
/////////////////////////////////////////////////////////////////////////////
DWORD CDownloadDlg::DownloadThread(LPVOID pvParam)
{
  CDownloadDlg* pcDlg = (CDownloadDlg*)pvParam;

  long lRet = CcifXTestDlg::s_pcDevice->Download(pcDlg->m_ulChannel,
                                                 pcDlg->m_ptMode->ulMode,
                                                 pcDlg->m_ptMode->csLastShortFilename.GetBuffer(0),
                                                 pcDlg->m_pabFileData,
                                                 pcDlg->m_ulFileSize,
                                                 DownloadProgress,
                                                 pcDlg);
  pcDlg->EnableWindow(TRUE);

  if(CIFX_NO_ERROR != lRet)
  {
    // Show last error
    CString csTemp;
    csTemp.Format(_T("0x%08X\r\n%s"),
                  lRet,
                  (LPCTSTR)CcifXTestDlg::s_pcDevice->GetErrorDescription(lRet));
    pcDlg->SetDlgItemText(IDC_LASTERR, csTemp);
  }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
/// Download button clicked event
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::OnBnClickedBtnDownload()
{
  CFile cFile;

  // Remove last error and clear progress bar
  SetDlgItemText(IDC_LASTERR, NULL);
  m_cDownloadProgressCtrl.SetPos(0);
  
  if(!cFile.Open(m_ptMode->csLastFullFilename, CFile::modeRead))
  {
    CString csTemp;
    csTemp.Format(_T("Error opening file '%s'. Error=%u"), (LPCTSTR)m_ptMode->csLastFullFilename, GetLastError());
    AfxMessageBox(csTemp);

  } else 
  {
    m_ulFileSize = (unsigned long)cFile.GetLength();
    
    if(NULL != m_pabFileData)
    {
      delete [] m_pabFileData;
      m_pabFileData = NULL;
    }

    m_pabFileData = new unsigned char[m_ulFileSize];

    if(NULL == m_pabFileData)
    {
      AfxMessageBox(_T("Error allocating memory for file!"));

    } else if(cFile.Read(m_pabFileData, m_ulFileSize) != m_ulFileSize)
    {
      AfxMessageBox(_T("Error reading file from Disk!"));
    } else
    {
      CreateThread(NULL,
                   0,
                   DownloadThread,
                   this,
                   0,
                   NULL);

      EnableWindow(FALSE);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////
/// No Cyclic thread function in this dialog
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::ThreadFunction()
{
  //no cyclic thread needed
}

/////////////////////////////////////////////////////////////////////////////
/// Event signalling device change in Main dialog
///   \param pcDevice NULL on device close
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::OnUpdateDevice(CCifXDeviceBase* pcDevice)
{
  if(NULL != pcDevice)
  {
    m_ulChannel = pcDevice->GetChannelNr();

    SetDlgItemInt(IDC_EDT_CHANNEL, m_ulChannel, FALSE);

    CEdit* pcEdit = (CEdit*)GetDlgItem(IDC_EDT_CHANNEL);
    pcEdit->SetReadOnly(!pcDevice->IsSystemDevice());
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Updates the progress bar and error cell
///   \param ulStep     Actual download step
///   \param ulMaxStep  Maximum steps
///   \param bFinished  != 0 if finished
///   \param lError     CIFX_NO_ERROR on successful download
/////////////////////////////////////////////////////////////////////////////
void CDownloadDlg::UpdateProgress(unsigned long ulStep, unsigned long ulMaxStep, char bFinished, long lError)
{
  m_cDownloadProgressCtrl.SetRange32(0, ulMaxStep);
  m_cDownloadProgressCtrl.SetPos(ulStep);

  if(bFinished)
  {
    EnableWindow(TRUE);
  }

  if(CIFX_NO_ERROR != lError)
  {
    // Show last error
   CString csTemp;
    csTemp.Format(_T("0x%08X\r\n%s"),
                  lError,
                  (LPCTSTR)CcifXTestDlg::s_pcDevice->GetErrorDescription(lError));
    SetDlgItemText(IDC_LASTERR, csTemp);
  }
}
